/**
* \file: filedb_writer.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/stat.h>
#include <stdio.h>
#include <errno.h>
#include <dirent.h>
#include <unistd.h>

#include "app_iface/filedb_writer.h"
#include "control/configuration.h"
#include "utils/logger.h"
#include "utils/helper.h"

#define DIR_MODE 0755

static const char *DB_BASE_PATH="/tmp/.automounter";
static const char *DEVICE_DB_SUBPATH="/device_db";
static const char *MOUNTPOINT_DB_SUBPATH="/mounted_partitions_db";

static void filedb_writer_rmdb(void);

static void filedb_writer_create_device_file(const char *device_fn, device_t *device);
static void filedb_writer_write_device_file_content(FILE *file, device_t *device);
static void filedb_writer_remove_device_file(const char *device_fn);

static void filedb_writer_create_device_directory(const char *device_fn);
static void filedb_writer_remove_device_directory(const char *device_fn);
static void filedb_writer_clear_device_directory(char *device_path);

static void filedb_writer_create_partition_file(const char *partition_fn, const char *device_fn, partition_t *partition);
static void filedb_writer_write_partition_file_content(FILE *file, partition_t *partition);
static void filedb_writer_remove_partition_file(const char *partition_fn, const char *device_fn);

static void filedb_writer_create_mounted_simlink(const char *partition_fn, const char *device_fn);
static void filedb_writer_remove_mounted_simlink(const char *partition_fn);

static void filedb_writer_create_device_fn(char *device_fn,  size_t buf_len,
		device_t *device);
static void filedb_writer_create_partition_fn(char *partition_fn,  size_t part_buf_len,char *device_fn,
		 size_t dev_buf_len, partition_t *partition);

//------------------------------------------ API members -------------------------------------------------
error_code_t filedb_writer_init(void)
{
	error_code_t result=RESULT_OK;

	char cmd[PATH_MAX+20]; //we want to place a command before a path

	//ensure that no dbpath directory is there anymore
	filedb_writer_rmdb();

	snprintf(cmd,PATH_MAX+20,"mkdir -p %s%s &> /dev/null",DB_BASE_PATH,DEVICE_DB_SUBPATH);
	logger_log_debug("FILEDB_WRITER - Try to create the device db path: %s%s",DB_BASE_PATH,DEVICE_DB_SUBPATH);
	if (system(cmd)!=0)
		result=RESULT_INVALID_ARGS;

	snprintf(cmd,PATH_MAX+20,"mkdir -p %s%s &> /dev/null",DB_BASE_PATH,MOUNTPOINT_DB_SUBPATH);
	logger_log_debug("FILEDB_WRITER - Try to create the device db path: %s%s",DB_BASE_PATH,MOUNTPOINT_DB_SUBPATH);
	if (system(cmd)!=0)
		result=RESULT_INVALID_ARGS;

	return result;
}

void filedb_writer_deinit(void)
{
	filedb_writer_rmdb();
}

void filedb_writer_signal_device_detected(device_t *device)
{
	char device_fn[PATH_MAX];
	filedb_writer_create_device_fn(device_fn, PATH_MAX, device);
	filedb_writer_create_device_directory(device_fn);
}

void filedb_writer_signal_device_handled(device_t *device)
{
	char device_fn[PATH_MAX];
	filedb_writer_create_device_fn(device_fn, PATH_MAX, device);
	filedb_writer_create_device_file(device_fn, device);
}

void filedb_writer_signal_device_invalid(device_t *device)
{
	char device_fn[PATH_MAX];
	filedb_writer_create_device_fn(device_fn, PATH_MAX, device);
	filedb_writer_remove_device_directory(device_fn);
	filedb_writer_remove_device_file(device_fn);
}

void filedb_writer_signal_partition_handled(partition_t *partition)
{
	char partition_fn[PATH_MAX];
	char device_fn[PATH_MAX];
	partition_state_t state;

	state=partition_get_state(partition);

	filedb_writer_create_partition_fn(partition_fn, PATH_MAX, device_fn, PATH_MAX, partition);
	filedb_writer_create_partition_file(partition_fn, device_fn, partition);
	if (state==PARTITION_MOUNTED)
		filedb_writer_create_mounted_simlink(partition_fn,device_fn);

	if (state==PARTITION_UNMOUNTED)
		filedb_writer_remove_mounted_simlink(partition_fn);
}

void filedb_writer_signal_partition_invalid(partition_t *partition)
{
	char partition_fn[PATH_MAX];
	char device_fn[PATH_MAX];
	filedb_writer_create_partition_fn(partition_fn, PATH_MAX,device_fn, PATH_MAX, partition);
	filedb_writer_remove_partition_file(partition_fn, device_fn);
	filedb_writer_remove_mounted_simlink(partition_fn);
}
//--------------------------------------------------------------------------------------------------------

//---------------------------------------------- internal members ----------------------------------------
static void filedb_writer_rmdb(void)
{
	char cmd[PATH_MAX+20];
	snprintf(cmd,PATH_MAX+20,"rm -r %s &> /dev/null",DB_BASE_PATH);
	if(system(cmd))
	{
	}
	logger_log_debug("FILEDB_WRITER - Removing DB directory: %s",DB_BASE_PATH);
}

static void filedb_writer_create_device_directory(const char *device_fn)
{
	char device_path[PATH_MAX];
	snprintf(device_path,PATH_MAX,"%s%s/device%s",DB_BASE_PATH,DEVICE_DB_SUBPATH,device_fn);

	if (mkdir(device_path,DIR_MODE)==-1)
	{
		if (errno!=EEXIST)
		{
			logger_log_error("FILEDB_WRITER - Unable to create following directory for the device db:\n");
			logger_log_error("-> %s\n",device_path);
		}
		else
			//directory exists, empty and reuse it
			filedb_writer_clear_device_directory(device_path);
	}
}

static void filedb_writer_create_device_fn(char *device_fn, size_t buf_len,
		device_t *device)
{
	strncpy(device_fn, device_get_id(device),buf_len);
	helper_strcrplc(device_fn, '/', '_',buf_len);
}

static void filedb_writer_create_partition_fn(char *partition_fn, size_t part_buf_len,char *device_fn,
		 size_t dev_buf_len, partition_t *partition)
{
	device_t *device;

	device=partition_get_device(partition);
	filedb_writer_create_device_fn(device_fn,dev_buf_len,device);

	strncpy(partition_fn, partition_get_id(partition),part_buf_len);
	helper_strcrplc(partition_fn, '/', '_',part_buf_len);
}

static void filedb_writer_clear_device_directory(char *device_path)
{
	DIR *dp;
	struct dirent *ep;

	dp = opendir (device_path);
	if (dp != NULL)
	{
		while ((ep = readdir (dp))!=NULL)
			unlink(ep->d_name);
	    (void) closedir (dp);
	}
}

static void filedb_writer_remove_device_directory(const char *device_fn)
{
	char device_path[PATH_MAX];
	snprintf(device_path,PATH_MAX,"%s%s/device%s",DB_BASE_PATH,DEVICE_DB_SUBPATH,device_fn);

	filedb_writer_clear_device_directory(device_path);
	if (rmdir(device_path)==-1)
	{
		logger_log_error("FILEDB_WRITER - Unable to remove following directory from device db:\n");
		logger_log_error("-> %s\n",device_path);
	}
}

static void filedb_writer_create_device_file(const char *device_fn, device_t *device)
{
	FILE *file;

	char device_path[PATH_MAX];
	snprintf(device_path,PATH_MAX,"%s%s/device%s.info",DB_BASE_PATH,DEVICE_DB_SUBPATH,device_fn);

	file=fopen(device_path,"w");

	if (file!=NULL)
	{
		filedb_writer_write_device_file_content(file,device);
		fclose(file);
	}
	else
	{
		logger_log_error("FILEDB_WRITER - Unable to create following device db file:\n");
		logger_log_error("-> %s\n",device_path);
	}
}

static void filedb_writer_remove_device_file(const char *device_fn)
{
	char device_path[PATH_MAX];
	snprintf(device_path,PATH_MAX,"%s%s/device%s.info",DB_BASE_PATH,DEVICE_DB_SUBPATH,device_fn);

	if (unlink(device_path)==-1)
	{
		logger_log_error("FILEDB_WRITER - Unable to remove following device db file:\n");
		logger_log_error("-> %s\n",device_path);
	}
}

static void filedb_writer_write_device_file_content(FILE *file, device_t *device)
{
	device_state_t state;
	state=device_get_state(device);

	fprintf(file, "INTERFACE_ID=%s\n", device_get_id(device));
	fprintf(file, "IDENTIFIER=%s\n",device_get_device_identifier(device));
	fprintf(file, "STATE=%s\n", device_get_state_str(state));
	fprintf(file, "PARTITION_CNT=%d\n", device_get_expected_partition_cnt(device));
	fprintf(file, "TYPE=%s\n", device_get_device_type(device));
	fprintf(file, "AM_DEVICE_HANDLER=%s\n", device_get_handler_id(device));
}

static void filedb_writer_create_partition_file(const char *partition_fn, const char *device_fn,
		partition_t *partition)
{
	FILE *file;
	char partition_path[PATH_MAX];
	snprintf(partition_path,PATH_MAX,"%s%s/device%s/partition%s.info",
			DB_BASE_PATH,DEVICE_DB_SUBPATH,device_fn,partition_fn);

	file=fopen(partition_path,"w");

	if (file!=NULL)
	{
		filedb_writer_write_partition_file_content(file,partition);
		fclose(file);
	}
	else
	{
		logger_log_error("FILEDB_WRITER - Unable to create following partition db file:\n");
		logger_log_error("-> %s\n",partition_path);
	}
}

static void filedb_writer_write_partition_file_content(FILE *file, partition_t *partition)
{
	partition_state_t state;

	state=partition_get_state(partition);

	fprintf(file, "INTERFACE_ID=%s\n", partition_get_id(partition));
	fprintf(file, "IDENTIFIER=%s\n",partition_get_partition_identifier(partition));
	fprintf(file, "STATE=%s\n", partition_get_state_str(state));

	if (state==PARTITION_MOUNTED)
		fprintf(file, "MOUNT_POINT=%s\n",partition_get_mountpoint(partition));
	else
		fprintf(file, "MOUNT_POINT=\n");
	fprintf(file, "MOUNT_SRC=%s\n",partition_get_mountsrc(partition));
	fprintf(file, "FILESYSTEM=%s\n",partition_get_mountfs(partition));
	fprintf(file, "PARTITION_NO=%d\n",partition_get_partition_number(partition));

	if (state==PARTITION_UNSUPPORTED)
		fprintf(file, "UNSUPPORTED_REASON=%s\n",partition_get_unsupported_reason_str(partition));
	else
		fprintf(file, "UNSUPPORTED_REASON=\n");

	if (partition_is_mounted_writable(partition))
		fprintf(file, "MOUNTED_WRITABLE=1\n");
	else
		fprintf(file, "MOUNTED_WRITABLE=0\n");
}

static void filedb_writer_remove_partition_file(const char *partition_fn, const char *device_fn)
{
	char partition_path[PATH_MAX];
	snprintf(partition_path,PATH_MAX,"%s%s/device%s/partition%s.info",
			DB_BASE_PATH,DEVICE_DB_SUBPATH,device_fn,partition_fn);

	if (unlink(partition_path)==-1)
	{
		logger_log_error("FILEDB_WRITER - Unable to remove following partition db file:\n");
		logger_log_error("-> %s\n",partition_path);
	}
}

static void filedb_writer_create_mounted_simlink(const char *partition_fn, const char *device_fn)
{
	char symlink_path[PATH_MAX];
	char symlink_pointer[PATH_MAX];

	snprintf(symlink_path,PATH_MAX,"%s%s/partition%s.info",
				DB_BASE_PATH,MOUNTPOINT_DB_SUBPATH,partition_fn);
	snprintf(symlink_pointer,PATH_MAX,"..%s/device%s/partition%s.info",DEVICE_DB_SUBPATH,device_fn,partition_fn);

	if (symlink(symlink_pointer, symlink_path))
	{
	}
}

static void filedb_writer_remove_mounted_simlink(const char *partition_fn)
{
	char symlink_path[PATH_MAX];
	snprintf(symlink_path,PATH_MAX,"%s%s/partition%s.info",
				DB_BASE_PATH,MOUNTPOINT_DB_SUBPATH,partition_fn);

	unlink(symlink_path);
}
//--------------------------------------------------------------------------------------------------------
